<?php

namespace ADP\BaseVersion\Includes\WP;

defined('ABSPATH') or exit;

/**
 * Object Cache API: WP_Object_Cache class
 *
 * @package WordPress
 * @subpackage Cache
 * @since 5.4.0
 */

/**
 * Core class that implements an object cache.
 *
 * The WordPress Object Cache is used to save on trips to the database. The
 * Object Cache stores all of the cache data to memory and makes the cache
 * contents available by using a key, which is used to name and later retrieve
 * the cache contents.
 *
 * The Object Cache can be replaced by other caching mechanisms by placing files
 * in the wp-content folder which is looked at in wp-settings. If that file
 * exists, then this file will not be included.
 *
 * @since 2.0.0
 */
class WpObjectCache
{

    /**
     * Holds the cached objects.
     *
     * @since 2.0.0
     * @var array
     */
    private $cache = array();

    /**
     * The amount of times the cache data was already stored in the cache.
     *
     * @since 2.5.0
     * @var int
     */
    public $cacheHits = 0;

    /**
     * Amount of times the cache did not have the request in cache.
     *
     * @since 2.0.0
     * @var int
     */
    public $cacheMisses = 0;

    /**
     * List of global cache groups.
     *
     * @since 3.0.0
     * @var array
     */
    protected $globalGroups = array();

    /**
     * The blog prefix to prepend to keys in non-global groups.
     *
     * @since 3.5.0
     * @var string
     */
    private $blogPrefix;

    /**
     * Holds the value of is_multisite().
     *
     * @since 3.5.0
     * @var bool
     */
    private $multisite;

    /**
     * Sets up object properties; PHP 5 style constructor.
     *
     * @since 2.0.8
     */
    public function __construct()
    {
        $this->multisite  = is_multisite();
        $this->blogPrefix = $this->multisite ? get_current_blog_id() . ':' : '';
    }

    /**
     * Makes private properties readable for backward compatibility.
     *
     * @param string $name Property to get.
     *
     * @return mixed Property.
     * @since 4.0.0
     *
     */
    public function __get($name)
    {
        return $this->$name;
    }

    /**
     * Makes private properties settable for backward compatibility.
     *
     * @param string $name Property to set.
     * @param mixed $value Property value.
     *
     * @return mixed Newly-set property.
     * @since 4.0.0
     *
     */
    public function __set($name, $value)
    {
        return $this->$name = $value;
    }

    /**
     * Makes private properties checkable for backward compatibility.
     *
     * @param string $name Property to check if set.
     *
     * @return bool Whether the property is set.
     * @since 4.0.0
     *
     */
    public function __isset($name)
    {
        return isset($this->$name);
    }

    /**
     * Makes private properties un-settable for backward compatibility.
     *
     * @param string $name Property to unset.
     *
     * @since 4.0.0
     *
     */
    public function __unset($name)
    {
        unset($this->$name);
    }

    /**
     * Adds data to the cache if it doesn't already exist.
     *
     * @param int|string $key What to call the contents in the cache.
     * @param mixed $data The contents to store in the cache.
     * @param string $group Optional. Where to group the cache contents. Default 'default'.
     * @param int $expire Optional. When to expire the cache contents. Default 0 (no expiration).
     *
     * @return bool True on success, false if cache key and group already exist.
     * @uses WP_Object_Cache::set()     Sets the data after the checking the cache
     *                                  contents existence.
     *
     * @since 2.0.0
     *
     * @uses WP_Object_Cache::_exists() Checks to see if the cache already has data.
     */
    public function add($key, $data, $group = 'default', $expire = 0)
    {
        if (wp_suspend_cache_addition()) {
            return false;
        }

        if (empty($group)) {
            $group = 'default';
        }

        $id = $key;
        if ($this->multisite && ! isset($this->globalGroups[$group])) {
            $id = $this->blogPrefix . $key;
        }

        if ($this->_exists($id, $group)) {
            return false;
        }

        return $this->set($key, $data, $group, (int)$expire);
    }

    /**
     * Sets the list of global cache groups.
     *
     * @param array $groups List of groups that are global.
     *
     * @since 3.0.0
     *
     */
    public function add_global_groups($groups)
    {
        $groups = (array)$groups;

        $groups             = array_fill_keys($groups, true);
        $this->globalGroups = array_merge($this->globalGroups, $groups);
    }

    /**
     * Decrements numeric cache item's value.
     *
     * @param int|string $key The cache key to decrement.
     * @param int $offset Optional. The amount by which to decrement the item's value. Default 1.
     * @param string $group Optional. The group the key is in. Default 'default'.
     *
     * @return int|false The item's new value on success, false on failure.
     * @since 3.3.0
     *
     */
    public function decr($key, $offset = 1, $group = 'default')
    {
        if (empty($group)) {
            $group = 'default';
        }

        if ($this->multisite && ! isset($this->globalGroups[$group])) {
            $key = $this->blogPrefix . $key;
        }

        if ( ! $this->_exists($key, $group)) {
            return false;
        }

        if ( ! is_numeric($this->cache[$group][$key])) {
            $this->cache[$group][$key] = 0;
        }

        $offset = (int)$offset;

        $this->cache[$group][$key] -= $offset;

        if ($this->cache[$group][$key] < 0) {
            $this->cache[$group][$key] = 0;
        }

        return $this->cache[$group][$key];
    }

    /**
     * Removes the contents of the cache key in the group.
     *
     * If the cache key does not exist in the group, then nothing will happen.
     *
     * @param int|string $key What the contents in the cache are called.
     * @param string $group Optional. Where the cache contents are grouped. Default 'default'.
     * @param bool $deprecated Optional. Unused. Default false.
     *
     * @return bool False if the contents weren't deleted and true on success.
     * @since 2.0.0
     *
     */
    public function delete($key, $group = 'default', $deprecated = false)
    {
        if (empty($group)) {
            $group = 'default';
        }

        if ($this->multisite && ! isset($this->globalGroups[$group])) {
            $key = $this->blogPrefix . $key;
        }

        if ( ! $this->_exists($key, $group)) {
            return false;
        }

        unset($this->cache[$group][$key]);

        return true;
    }

    /**
     * Clears the object cache of all data.
     *
     * @return true Always returns true.
     * @since 2.0.0
     *
     */
    public function flush()
    {
        $this->cache = array();

        return true;
    }

    /**
     * Retrieves the cache contents, if it exists.
     *
     * The contents will be first attempted to be retrieved by searching by the
     * key in the cache group. If the cache is hit (success) then the contents
     * are returned.
     *
     * On failure, the number of cache misses will be incremented.
     *
     * @param int|string $key What the contents in the cache are called.
     * @param string $group Optional. Where the cache contents are grouped. Default 'default'.
     * @param bool $force Optional. Unused. Whether to force a refetch rather than relying on the local
     *                           cache. Default false.
     * @param bool $found Optional. Whether the key was found in the cache (passed by reference).
     *                           Disambiguates a return of false, a storable value. Default null.
     *
     * @return mixed|false The cache contents on success, false on failure to retrieve contents.
     * @since 2.0.0
     *
     */
    public function get($key, $group = 'default', $force = false, &$found = null)
    {
        if (empty($group)) {
            $group = 'default';
        }

        if ($this->multisite && ! isset($this->globalGroups[$group])) {
            $key = $this->blogPrefix . $key;
        }

        if ($this->_exists($key, $group)) {
            $found           = true;
            $this->cacheHits += 1;
            if (is_object($this->cache[$group][$key])) {
                return clone $this->cache[$group][$key];
            } else {
                return $this->cache[$group][$key];
            }
        }

        $found             = false;
        $this->cacheMisses += 1;

        return false;
    }

    /**
     * Increments numeric cache item's value.
     *
     * @param int|string $key The cache key to increment
     * @param int $offset Optional. The amount by which to increment the item's value. Default 1.
     * @param string $group Optional. The group the key is in. Default 'default'.
     *
     * @return int|false The item's new value on success, false on failure.
     * @since 3.3.0
     *
     */
    public function incr($key, $offset = 1, $group = 'default')
    {
        if (empty($group)) {
            $group = 'default';
        }

        if ($this->multisite && ! isset($this->globalGroups[$group])) {
            $key = $this->blogPrefix . $key;
        }

        if ( ! $this->_exists($key, $group)) {
            return false;
        }

        if ( ! is_numeric($this->cache[$group][$key])) {
            $this->cache[$group][$key] = 0;
        }

        $offset = (int)$offset;

        $this->cache[$group][$key] += $offset;

        if ($this->cache[$group][$key] < 0) {
            $this->cache[$group][$key] = 0;
        }

        return $this->cache[$group][$key];
    }

    /**
     * Replaces the contents in the cache, if contents already exist.
     *
     * @param int|string $key What to call the contents in the cache.
     * @param mixed $data The contents to store in the cache.
     * @param string $group Optional. Where to group the cache contents. Default 'default'.
     * @param int $expire Optional. When to expire the cache contents. Default 0 (no expiration).
     *
     * @return bool False if not exists, true if contents were replaced.
     * @see WP_Object_Cache::set()
     *
     * @since 2.0.0
     *
     */
    public function replace($key, $data, $group = 'default', $expire = 0)
    {
        if (empty($group)) {
            $group = 'default';
        }

        $id = $key;
        if ($this->multisite && ! isset($this->globalGroups[$group])) {
            $id = $this->blogPrefix . $key;
        }

        if ( ! $this->_exists($id, $group)) {
            return false;
        }

        return $this->set($key, $data, $group, (int)$expire);
    }

    /**
     * Resets cache keys.
     *
     * @since 3.0.0
     *
     * @deprecated 3.5.0 Use switch_to_blog()
     * @see switch_to_blog()
     */
    public function reset()
    {
        _deprecated_function(__FUNCTION__, '3.5.0', 'switch_to_blog()');

        // Clear out non-global caches since the blog ID has changed.
        foreach (array_keys($this->cache) as $group) {
            if ( ! isset($this->globalGroups[$group])) {
                unset($this->cache[$group]);
            }
        }
    }

    /**
     * Sets the data contents into the cache.
     *
     * The cache contents are grouped by the $group parameter followed by the
     * $key. This allows for duplicate ids in unique groups. Therefore, naming of
     * the group should be used with care and should follow normal function
     * naming guidelines outside of core WordPress usage.
     *
     * The $expire parameter is not used, because the cache will automatically
     * expire for each time a page is accessed and PHP finishes. The method is
     * more for cache plugins which use files.
     *
     * @param int|string $key What to call the contents in the cache.
     * @param mixed $data The contents to store in the cache.
     * @param string $group Optional. Where to group the cache contents. Default 'default'.
     * @param int $expire Not Used.
     *
     * @return true Always returns true.
     * @since 2.0.0
     *
     */
    public function set($key, $data, $group = 'default', $expire = 0)
    {
        if (empty($group)) {
            $group = 'default';
        }

        if ($this->multisite && ! isset($this->globalGroups[$group])) {
            $key = $this->blogPrefix . $key;
        }

        if (is_object($data)) {
            $data = clone $data;
        }

        $this->cache[$group][$key] = $data;

        return true;
    }

    /**
     * Echoes the stats of the caching.
     *
     * Gives the cache hits, and cache misses. Also prints every cached group,
     * key and the data.
     *
     * @since 2.0.0
     */
    public function stats()
    {
        echo '<p>';
        echo "<strong>Cache Hits:</strong> {$this->cacheHits}<br />";
        echo "<strong>Cache Misses:</strong> {$this->cacheMisses}<br />";
        echo '</p>';
        echo '<ul>';
        foreach ($this->cache as $group => $cache) {
            echo '<li><strong>Group:</strong> ' . esc_html($group) . ' - ( ' . number_format(strlen(serialize($cache)) / KB_IN_BYTES,
                    2) . 'k )</li>';
        }
        echo '</ul>';
    }

    /**
     * Switches the internal blog ID.
     *
     * This changes the blog ID used to create keys in blog specific groups.
     *
     * @param int $blogId Blog ID.
     *
     * @since 3.5.0
     *
     */
    public function switch_to_blog($blogId)
    {
        $blogId           = (int)$blogId;
        $this->blogPrefix = $this->multisite ? $blogId . ':' : '';
    }

    /**
     * Serves as a utility function to determine whether a key exists in the cache.
     *
     * @param int|string $key Cache key to check for existence.
     * @param string $group Cache group for the key existence check.
     *
     * @return bool Whether the key exists in the cache for the given group.
     * @since 3.4.0
     *
     */
    protected function _exists($key, $group)
    {
        return isset($this->cache[$group]) && (isset($this->cache[$group][$key]) || array_key_exists($key,
                    $this->cache[$group]));
    }
}
